set scan off
Create or Replace Package Body CLEAN_Address_APP_UE
IS
gv_Package_Body_Version  CONSTANT VARCHAR2(30) := '3.5.0.0';
gv_Package_Body_Date     CONSTANT VARCHAR2(30) := '1-Apr-2008';
/*****************************************************************************************
 *
 *  Package Name    :   CLEAN_Address_APP_UE
 *
 *  Description     :   User Exit extensions for the CLEAN_Address_APP package
 *
 *                      This package is used to extend or enhance the data quality
 *                        procedures in CLEAN_Address_APP through the use of pre- and
 *                        post- transaction user exit procedures.
 *
 *  Version         :   3.5.0.0  Apr 1, 2008
 *
 * ---------------------------------------------------------------------------------------
 *   Customer has the right to modify this package so long as original copyright remains
 * ---------------------------------------------------------------------------------------
 * History:
 * MM/DD/YYYY User    Comment
 * ---------- ------- --------------------------------------------------------------------
 *****************************************************************************************
 * Copyright (c) 2004-2005 Runner Technologies, Inc.  All Rights Reserved.
 * www.RunnerTechnologies.com   (877)784-0003  561-395-9322
 *****************************************************************************************/
-- Set the batch mode for decoding in Pre and Post user exits
gb_batch_mode       BOOLEAN := FALSE;
-----------------------------------------------------------------------------------------
-- PRIVATE PROCEDURES
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Replace_All
 *
 *  Scope           :   Private
 *
 *  Description     :   Replace all occurances of a string in the address elements.
 *                      Do replacements based on EXACT, UPPER, LOWER, INITCAP
 ******************************************************************************************/
Procedure Replace_All (
                     f_address_rec                  IN OUT NOCOPY CLEAN_Address.Address_rec
                    ,fv_from                        IN     VARCHAR2
                    ,fv_to                          IN     VARCHAR2
                    )
IS
BEGIN
  --
  -- Replace exact text AS-IS
  --
  CLEAN_Address.Replace_Component(f_address_rec.address_line_1, fv_from, fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_2, fv_from, fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_3, fv_from, fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.Suite,          fv_from, fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address,        fv_from, fv_to, 30);
  --
  -- Replace text in UPPER case
  --
  CLEAN_Address.Replace_Component(f_address_rec.address_line_1, UPPER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_2, UPPER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_3, UPPER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.Suite,          UPPER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address,        UPPER(fv_from), fv_to, 30);
  --
  -- Replace text in LOWER case
  --
  CLEAN_Address.Replace_Component(f_address_rec.address_line_1, LOWER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_2, LOWER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_3, LOWER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.Suite,          LOWER(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address,        LOWER(fv_from), fv_to, 30);
  --
  -- Replace text in INITCAP case
  --
  CLEAN_Address.Replace_Component(f_address_rec.address_line_1, INITCAP(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_2, INITCAP(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address_line_3, INITCAP(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.Suite,          INITCAP(fv_from), fv_to, 30);
  CLEAN_Address.Replace_Component(f_address_rec.address,        INITCAP(fv_from), fv_to, 30);
END Replace_All;
-----------------------------------------------------------------------------------------
-- PUBLIC PROCEDURES
-----------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------
-- ADDRESS USER EXITS
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Pre_Address_Verify
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE an address is verified by CLEAN_Address
 *                      * To skip address verification for this address, set fb_verified=TRUE
 *
 * Use this procedure to add any custom business rules, or bypass address verification:
 * - Bypass address verification for certain address types (ie. Campus Addresses)
 * - Different capitalization and standardization rules for address types (Alumni spell out suffix)
 * - Special shortcut codes for addresses (ie. ' OBT ' = ' Orange Blossom Trail ')
 *
 * NOTE: You can set GLOBAL defaults in the package body constructor at the bottom of the body
 ******************************************************************************************/
PROCEDURE Pre_Address_Verify (
                     f_address_rec                  IN OUT CLEAN_Address.Address_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_address_type                IN     VARCHAR2
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
/* BEGIN EXAMPLES
  --
  -- Cursor to get the database name for dynamically enabling CLEAN_Address in certain databases
  -- I.e. If you only want CLEAN_Address to verify addresses in DEV and PROD
  --
  cursor c_dbname IS
    select upper(substr(global_name,1,decode(instr(global_name,'.'),0,length(global_name)+1,instr(global_name,'.')-1))) dbname
    from   global_name;
  lv_dbname      VARCHAR2(50);
END EXAMPLES */
BEGIN
/* BEGIN EXAMPLES - uncomment and edit any blocks of code that apply for your specific business rules or standards
  --
  -- Enable address verification only in PRODUCTION database, otherwise skip verification
  -- Read the database name from the GLOBAL_NAME table
  --
  OPEN  c_dbname;
  FETCH c_dbname
  INTO  lv_dbname;
  CLOSE c_dbname;
  -- Example: Only allow address verification in DEV and PROD
  if lv_dbname NOT IN ('DEV','PROD') then
    fb_verified := TRUE;
    return;
  end if;
  --
  -- Don't verify Campus Addresses (C) or Dorm Rooms (D)
  --
  if upper(fv_address_type) in ('C', 'D') then
    fb_verified := TRUE;
  end if;
  --
  -- Check for a forced address containing ! at the end.
  -- Strip off the ! before saving and skip verification.
  -- Don't allow WWW users to override addresses.
  --
  if   (f_address_rec.address_line_1 like '%!'
     or f_address_rec.address_line_2 like '%!'
     or f_address_rec.address_line_3 like '%!')
     and USER not like 'WWW%'
     then
    f_address_rec.address_line_1 := replace(f_address_rec.address_line_1,'!', '');
    f_address_rec.address_line_2 := replace(f_address_rec.address_line_2,'!', '');
    f_address_rec.address_line_3 := replace(f_address_rec.address_line_3,'!', '');
    fb_verified := TRUE;
  end if;
END EXAMPLES */
  /* COMMENT OUT THE NEXT SECTION TO VERIFY INTERNATIONAL ADDRESSES */
  --
  -- Only validate US addresses through USPS engine, and set county to NULL for international addresses
  --
  -- Next line is default US Setting - comment to show US for USPS addresses
  if NVL(upper(f_address_rec.country_code),'USA') NOT IN ('US','USA','157')
  -- Next line is default Canada Setting - uncomment to only verify Canadian addresses
  --if NVL(upper(f_address_rec.country_code),'CA') NOT IN ('CA','CAN','CANADA')
     -- Country_Name is set in the CLEAN_Address_Banner package for international addresses
     or f_address_rec.country_name IS NOT NULL
     then
    fb_verified := TRUE;
    f_address_rec.county_name := ' ';
    f_address_rec.Error_Code  := 'SKIP';
  end if;
  --
  -- Make sure we have an address line before verification
  --
  if f_address_rec.address_line_1||f_address_rec.address_line_2||f_address_rec.address_line_3 IS NULL then
    -- don't try to verify without street information
    fb_verified := TRUE;
    f_address_rec.Error_Code  := 'SKIP';
    return;
  end if;
  --
  -- Skip addresses with a SKIP address type
  --
  if fv_address_type in ('SKIP') then
    fb_verified := TRUE;
    f_address_rec.Error_Code  := 'SKIP';
  end if;
  --
  -- Set the address suggestion count to 20 when an address is unverified - only in real-time mode
  --
  if NOT gb_batch_mode then
    CLEAN_Address_APP.Set_Error_Suggest_Count(20);
  end if;
END Pre_Address_Verify;
/******************************************************************************************
 *  Procedure Name  :   Post_Address_Verify
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing immediately AFTER an address is verified
 *                      by CLEAN_Address
 *                      NOTE: This is called BEFORE the built-in business rules for Application
 *                            are applied.
 *                      * To skip additional business rules from firing, set fb_verified=TRUE
 ******************************************************************************************/
PROCEDURE Post_Address_Verify (
                     f_address_rec                  IN OUT CLEAN_Address.Address_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_address_type                IN     VARCHAR2
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
  --
  -- Set the delivered defaults for all addresses: (comment out or delete code to remove rule)
  -- - Expand the street suffix
  -- - Put Apt/Suite on same line as address if it will fit, or on line after if it doesn't fit
  -- - Expand the directional only if it's N S E W
  --
/* EXAMPLE
  if CLEAN_Address_APP.Get_Object_Name like '%MY_ADDRESS_PAGE' then
    CLEAN_Address_APP.Set_Expand_Street_Suffix(FALSE);
  else
    NULL;
  end if;
  */
/* EXAMPLES - comment and/or edit any blocks of code that apply for your specific business rules or standards */
  --
  -- Use different rules for Formal addresses (FORMAL)
  --
  if upper(fv_address_type) in ('FORMAL') then
    --
    CLEAN_Address_APP.Set_Expand_Street_Suffix(TRUE);
    --
    -- Expand the street directional N S E W -> North, South, East, West
    if    f_address_rec.parsed_pre_direction  IN ('N','S','E','W')
       or f_address_rec.parsed_post_direction IN ('N','S','E','W') then
      --
      if NVL(f_address_rec.address_line_1,'x') = NVL(f_address_rec.address,'y') then
        CLEAN_Address.Expand_Street_Directional(f_address_rec.address_line_1, CLEAN_Address_APP.Get_Max_Address_Length);
      end if;
      if NVL(f_address_rec.address_line_2,'x') = NVL(f_address_rec.address,'y') then
        CLEAN_Address.Expand_Street_Directional(f_address_rec.address_line_2, CLEAN_Address_APP.Get_Max_Address_Length);
      end if;
      if NVL(f_address_rec.address_line_3,'x') = NVL(f_address_rec.address,'y') then
        CLEAN_Address.Expand_Street_Directional(f_address_rec.address_line_3, CLEAN_Address_APP.Get_Max_Address_Length);
      end if;
      CLEAN_Address.Expand_Street_Directional(f_address_rec.address,          CLEAN_Address_APP.Get_Max_Address_Length);
    end if;
    --
    CLEAN_Address_APP.Set_Suite_Apt_Before_Street(FALSE);
    CLEAN_Address_APP.Set_Append_Suite_To_Street(TRUE);
    --
    -- Replace out other address abbreviations based on standards
    --
    --Replace_All(f_address_rec, 'PO Box',        'Post Office Box');
    --Replace_All(f_address_rec, 'RR',            'Rural Route');
    --Replace_All(f_address_rec, 'Apt',           'Apartment');
    Replace_All(f_address_rec, 'Ste',           'Suite');
  else
    --
    -- Use Default Standards for other address types
    --
    CLEAN_Address_APP.Set_Expand_Street_Suffix(FALSE);
    CLEAN_Address_APP.Set_Suite_Apt_Before_Street(FALSE);
    CLEAN_Address_APP.Set_Append_Suite_To_Street(TRUE);
  end if;
  /* End Examples */
  --
  -- Use NULL for the country code if US or Canada
  --
  -- Next line is default US Setting - comment to show US for USPS addresses
  if NVL(upper(f_address_rec.country_code),'USA') IN ('US','USA') then
  -- Next line is default Canada Setting - uncomment to show NULL for Canadian addresses
  --if NVL(upper(f_address_rec.country_code),'CA') IN ('CA','CAN','CANADA') then
    f_address_rec.country_code := 'USA';
  end if;
END Post_Address_Verify;
/******************************************************************************************
 *  Procedure Name  :   Post_Address_Process
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing AFTER an address is verified by CLEAN_Address
 *                      NOTE: This is called just before the Verify_Address_Record procedure
 *                            returns, AFTER the built-in business rules for Application
 ******************************************************************************************/
PROCEDURE Post_Address_Process (
                     f_address_rec                  IN OUT CLEAN_Address.Address_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_address_type                IN     VARCHAR2
                    )
IS
BEGIN
  NULL;
END Post_Address_Process;
/******************************************************************************************
 *  Procedure Name  :   Pre_Batch_Address_Verify
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE the batch address verify procedure is run
 *
 * Use this procedure to add any custom business rules or set global parameters:
 * - Primarily to set or unset the gb_Batch_Update_Existing to update existing or create new addresses
 ******************************************************************************************/
PROCEDURE Pre_Batch_Address_Verify
IS
BEGIN
/* EXAMPLE
  --
  -- Always update existing records when updating in batch mode
  -- - If set to TRUE existing address record will be updated in batch mode.
  -- - If set to FALSE, a new record will be created if anything changes in the address.
  -- NOTE: This could easily double the size of the SPRADDR table if set FALSE, since most
  --       addresses are not entered in the USPS standards. Please take this into consideration.
  CLEAN_Address_APP.Set_Batch_Update_Existing(TRUE);
  */
  -- Set global batch indicator to TRUE
  gb_batch_mode := TRUE;
END Pre_Batch_Address_Verify;
/******************************************************************************************
 *  Procedure Name  :   Post_Batch_Address_Verify
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing AFTER the batch address verify procedure is run
 *
 *  Use this procedure to email or save any batch update results
 ******************************************************************************************/
PROCEDURE Post_Batch_Address_Verify
IS
BEGIN
  -- Set global batch indicator to FALSE at the end of batch processing
  gb_batch_mode := FALSE;
END Post_Batch_Address_Verify;
-----------------------------------------------------------------------------------------
-- TELEPHONE USER EXITS
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Pre_Telephone_Check
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE a telephone is checked by CLEAN_Address
 *                      * To skip telephone checking, set fb_verified=TRUE
 *
 * Use this procedure to add any custom business rules, or bypass telephone checking:
 * - Bypass telephone checking completely for certain telephone types (ie. Campus Addresses)
 * - Add custom telephone checking logic for campus extensions
 ******************************************************************************************/
PROCEDURE Pre_Telephone_Check (
                     f_telephone_rec                IN OUT CLEAN_Address.Phone_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_telephone_type              IN     VARCHAR2
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
/* EXAMPLES
  --
  -- Don't verify cell phones
  --
  if upper(fv_telephone_type) in ('CELL') then
    fb_verified := TRUE;
  end if;
  */
  --
  -- Override the default Telephone format mask
  --
  --f_telephone_rec.Phone_Number_Output_Format := '999.999.9999';
  --f_telephone_rec.Phone_Number_Output_Format := '999/999.9999';
  f_telephone_rec.Phone_Number_Output_Format := '999-999-9999';
END Pre_Telephone_Check;
/******************************************************************************************
 *  Procedure Name  :   Post_Telephone_Check
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing immediately AFTER a telephone is verified
 *                      by CLEAN_Address
 *                      NOTE: This is called BEFORE the built-in business rules for Application
 *                            are applied.
 *                      * To skip additional business rules from firing, set fb_verified=TRUE
 ******************************************************************************************/
PROCEDURE Post_Telephone_Check (
                     f_telephone_rec                IN OUT CLEAN_Address.Phone_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_telephone_type              IN     VARCHAR2
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
  NULL;
END Post_Telephone_Check;
/******************************************************************************************
 *  Procedure Name  :   Post_Telephone_Process
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing AFTER a telephone is checked by CLEAN_Address
 *                      NOTE: This is called just before the Check_Telephone_Record procedure
 *                            returns, AFTER the built-in business rules for Application
 ******************************************************************************************/
PROCEDURE Post_Telephone_Process (
                     f_telephone_rec                IN OUT CLEAN_Address.Phone_rec
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_telephone_type              IN     VARCHAR2
                    )
IS
BEGIN
  NULL;
END Post_Telephone_Process;
/******************************************************************************************
 *  Procedure Name  :   Pre_Batch_Telephone_Check
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE the batch telephone check procedure is run
 ******************************************************************************************/
PROCEDURE Pre_Batch_Telephone_Check
IS
BEGIN
  gb_batch_mode := TRUE;
END Pre_Batch_Telephone_Check;
/******************************************************************************************
 *  Procedure Name  :   Post_Batch_Telephone_Check
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing AFTER the batch telephone check procedure is run
 *
 *  Use this procedure to email or save any batch update results
 ******************************************************************************************/
PROCEDURE Post_Batch_Telephone_Check
IS
BEGIN
  gb_batch_mode := FALSE;
END Post_Batch_Telephone_Check;
-----------------------------------------------------------------------------------------
--
-- NAME STANDARDIZE USER EXITS
--
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Pre_Name_Standardize
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE a name is standardized by CLEAN_Address
 *                      * To skip telephone checking, set fb_verified=TRUE
 *
 * Use this procedure to add any custom business rules, or bypass name standardization:
 * - Bypass name standardization completely for certain name types
 ******************************************************************************************/
PROCEDURE Pre_Name_Standardize (
                     f_name_rec                     IN OUT CLEAN_Address.Name_rec
                    ,fv_name_error_text             IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_name_type_code              IN     VARCHAR2
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
  NULL;
  -- Disable Name Standardization
  --fb_verified := TRUE;
END Pre_Name_Standardize;
/******************************************************************************************
 *  Procedure Name  :   Post_Name_Standardize
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing immediately AFTER a name is standardized
 *                      by CLEAN_Address
 *                      * To skip additional business rules from firing, set fb_verified=TRUE
 *
 * Use this procedure to add any custom name standardizations
 ******************************************************************************************/
PROCEDURE Post_Name_Standardize (
                     f_name_rec                     IN OUT CLEAN_Address.Name_rec
                    ,fv_name_error_text             IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_name_type_code              IN     VARCHAR2
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
/* EXAMPLES
  --
  -- Always put a period after certain prefixes
  --
  if f_name_rec.name_prefix in ('Mr', 'Mrs', 'Ms', 'Dr') then
    f_name_rec.name_prefix := f_name_rec.name_prefix||'.';
  end if;
  --
  -- Always put a period after certain suffixes
  --
  if instr(' '||f_name_rec.name_suffix||' ', ' Jr ') > 0 then
    CLEAN_Address.Replace_Component(f_name_rec.name_suffix, 'Jr', 'Jr.');
  end if;
  */
  -- Add prefix always to firstname
  if f_name_rec.name_prefix IS NOT NULL then
    f_name_rec.first_name := f_name_rec.name_prefix||' '||f_name_rec.first_name;
  end if;
  -- Add suffix always to lastname
  if f_name_rec.name_suffix IS NOT NULL then
    f_name_rec.last_name := f_name_rec.last_name||' '||f_name_rec.name_suffix;
  end if;
END Post_Name_Standardize;
/******************************************************************************************
 *  Procedure Name  :   Pre_Batch_Name_Standardize
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE the batch name standardize procedure is run
 ******************************************************************************************/
PROCEDURE Pre_Batch_Name_Standardize
IS
BEGIN
  NULL;
END Pre_Batch_Name_Standardize;
/******************************************************************************************
 *  Procedure Name  :   Post_Batch_Name_Standardize
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing AFTER the batch name standardize procedure is run
 *
 *  Use this procedure to email or save any batch update results
 ******************************************************************************************/
PROCEDURE Post_Batch_Name_Standardize
IS
BEGIN
  NULL;
END Post_Batch_Name_Standardize;
-----------------------------------------------------------------------------------------
--
-- EMAIL VALIDATION USER EXITS
--
-----------------------------------------------------------------------------------------
/******************************************************************************************
 *  Procedure Name  :   Pre_Email_Validate
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE an email is validated by CLEAN_Address
 *                      * To skip telephone checking, set fb_verified=TRUE
 *
 * Use this procedure to add any custom business rules, or bypass email validation
 * - Bypass email standardization completely for certain email types
 ******************************************************************************************/
PROCEDURE Pre_Email_Validate (
                     f_email_rec                    IN OUT CLEAN_Address.Email_rec
                    ,fv_email_error_text            IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_email_type_code             IN     VARCHAR2
                    ,fb_dns_lookup                  IN OUT BOOLEAN
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
  NULL;
  -- Disable Email Validation
  --fb_verified := TRUE;
END Pre_Email_Validate;
/******************************************************************************************
 *  Procedure Name  :   Post_Email_Validate
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing immediately AFTER an email is validated
 *                      by CLEAN_Address
 *                      * To skip additional business rules from firing, set fb_verified=TRUE
 *
 * Use this procedure to add any custom email standardizations
 ******************************************************************************************/
PROCEDURE Post_Email_Validate (
                     f_email_rec                    IN OUT CLEAN_Address.Email_rec
                    ,fv_email_error_text            IN OUT VARCHAR2
                    ,fv_plsql_error                 IN OUT VARCHAR2
                    ,fv_email_type_code             IN     VARCHAR2
                    ,fb_dns_lookup                  IN OUT BOOLEAN
                    ,fb_verified                    IN OUT BOOLEAN
                    )
IS
BEGIN
  NULL;
END Post_Email_Validate;
/******************************************************************************************
 *  Procedure Name  :   Pre_Batch_Email_Validate
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing BEFORE the batch email validation procedure is run
 ******************************************************************************************/
PROCEDURE Pre_Batch_Email_Validate
IS
BEGIN
  NULL;
END Pre_Batch_Email_Validate;
/******************************************************************************************
 *  Procedure Name  :   Post_Batch_Email_Validate
 *
 *  Scope           :   PUBLIC
 *
 *  Description     :   User exit processing AFTER the batch email validation procedure is run
 *
 *  Use this procedure to email or save any batch update results
 ******************************************************************************************/
PROCEDURE Post_Batch_Email_Validate
IS
BEGIN
  NULL;
END Post_Batch_Email_Validate;
-----------------------------------------------------------------------------------------
--
-----------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------
-- DEFAULT CONSTRUCTOR BLOCK
-- - Use this global constructor to set the global parameters for the CLEAN_Address_APP package
-- - This will set the parameter defaults
-----------------------------------------------------------------------------------------
BEGIN
  -- Specify a longer timeout value when communicating with the CLEAN_Address server
  CLEAN_Address.Set_Address_Pipe_Name(fi_pipe_timeout_seconds => 12);
  --
  -- Set gv_Local_Country to the Banner specific Nation Code for the predominant country
  -- DEFAULT: US
  CLEAN_Address_APP.Set_Local_Country('USA');
  --
  -- Set gn_Max_Address_Length to limit size of address field
  CLEAN_Address_APP.Set_Max_Address_Length(35);
  --
/* EXAMPLES (see note about about DEFAULT CONSTRUCTOR BLOCK)
  --
  -- Set gb_expand_street_suffix to TRUE to expand the street suffix (Ave -> Avenue) if it will
  -- fit on one line.  If set to FALSE, the USPS CASS standardization rules will be used.
  -- DEFAULT: FALSE
  CLEAN_Address_APP.Set_Expand_Street_Suffix(FALSE);
  --
  -- Set gb_suite_apt_before_street to TRUE to put the suite/apartment line before the address line
  -- in cases where the the suite/apartment does not fit on one line.  If set to FALSE, the suite/apt
  -- will appear on the line after the street address line.
  -- DEFAULT: FALSE
  CLEAN_Address_APP.Set_Suite_Apt_Before_Street(FALSE);
  --
  -- Set gb_append_suite_to_street to TRUE to always "try" to append the suite/apartment line
  -- on the same line as the the address line - only when it will fit.  If set to FALSE, the
  -- suite/apt will be on a separate line if there a free line available
  -- DEFAULT: TRUE
  CLEAN_Address_APP.Set_Append_Suite_To_Street(TRUE);
  --
  -- Set gb_use_ZIP_Plus4 to TRUE to append the ZIP+4 digits after the ZIP Code in the
  -- SPRADDR_ZIP field.  If set to FALSE, only the 5-digit ZIP Code will be used
  -- DEFAULT: TRUE
  CLEAN_Address_APP.Set_Use_ZIP_Plus4(TRUE);
  --
  -- Set gb_batch_update_existing to TRUE to update the existing address record when running
  -- in batch mode.  If set to FALSE, a new record will be created if anything changes in the
  -- address.  This could easily double the size of the SPRADDR table if set FALSE, since most
  -- addresses are not entered in the USPS standards. Please take this into consideration.
  -- DEFAULT: TRUE
  CLEAN_Address_APP.Set_Batch_Update_Existing(TRUE);
  --
  -- Set gn_telephone_distance_limit to the number of miles threshhold to allow before an
  -- alert message is displayed to the user.  This is the distance between the telephone
  -- wire center (area code and prefix = first 6 digits) and the 5-digit ZIP Code
  -- DEFAULT: 100
  CLEAN_Address_APP.Set_Telephone_Distance_Limit(100);
  --
  -- Set gb_Upper_Case_Results to upper case all address results
  -- DEFAULT: FALSE
  CLEAN_Address_APP.Set_Upper_Case_Results(FALSE);
  --
  -- Set gb_Ignore_Error to TRUE to allow Web Self-Service to save data even with an error code
  -- NOTE: The error code will still be stored on the record but allow the transaction to continue
  -- The functions Set_Ignore_Error and Get_Ignore_Error are used for this variable
  -- DEFAULT: FALSE
  CLEAN_Address_APP.Set_Ignore_Error(FALSE);
  --
  -- Set gb_Skip_International TRUE to not process international address records for interactive processing
  -- This allows more control on the purchased Web Service transactions.
  -- NOTE: User should pass this parameter into the batch procedure rather than setting directly
  -- DEFAULT: TRUE
  CLEAN_Address_APP.Set_Skip_International(TRUE);
  --
  */
END CLEAN_Address_APP_UE;

/

set scan on
